' Ten kod pochodzi z ksiki "ASP.NET 2.0. Gotowe rozwizania" autorstwa
' Imara Spaanjaarsa, Paula Wiltona and Shawna Livermore, wydanej przez Wrox.
' Polska edycja: Wydawnictwo Helion.
' Copyright 2006 by Wiley Publishing Inc.
' Informacja o tej ksice jest dostpna na stronie www.wrox.com.  
' Odwied p2p.wrox.com aby podyskutowa o tym kodzie na forach Wrox.

Imports System.Collections.Generic

''' <summary>
''' Klasa BugManager jest uywana do tworzenia, edycji i usuwania poszczeglnych bdw oraz pracy z listami bdw. Pozwala rwnie sortowa bedy na licie za pomoc swojej klasy wewntrznej <see cref="BugManager.BugComparer"/>.
''' </summary>
Public Class BugManager

#Region "Zmienne prywatne"

  Dim _theBugList As List(Of Bug) ' Lista bdw przechowujca wszystkie bdy  
  Dim _memberId As Guid = Guid.Empty

#End Region

#Region "Konstruktor(y)"

  ''' <summary>
  ''' Konstruktor, ktry przyjmuje jako argument identyfikator biecego czonka zespou.
  ''' </summary>
  Public Sub New(ByVal memberId As Guid)
    _memberId = memberId
  End Sub

#End Region

#Region "Waciwoci publiczne"

  ''' <summary>
  ''' Zwraca liczb obiektw <see cref="Bug">Bugs</see> w BugList.
  ''' </summary>
  ''' <seealso cref="Bug"/>
  Public ReadOnly Property Count() As Integer
    Get
      If _theBugList IsNot Nothing Then
        Return _theBugList.Count
      Else
        Return 0
      End If
    End Get
  End Property

  ''' <summary>
  ''' Pobiera lub ustawia identyfikator czonka zespou korzystajcego obecnie z klasy BugManager.
  ''' Ta waciwo jest uywana do sprawdzania praw dostpu.
  ''' </summary>
  Public ReadOnly Property MemberId() As Guid
    Get
      Return _memberId
    End Get
  End Property

#End Region

#Region "Metody publiczne"

  ''' <overloads>Zwraca silnie typowany obiekt BugList zawierajcy wszystkie bdy.</overloads>
  ''' <summary>
  ''' Zwraca silnie typowan list wszystkich bdw, posortowan wg Id w kolejnoci malejcej.
  ''' Jeli chcesz w sposb jawny uporzdkowac list, wywoaj przecion wersj GetBugList(String, SearchCriteria).
  ''' </summary>
  ''' <param name="searchCriteria">Ustala kryteria wyszukiwania zastosowane w tej licie.</param>
  ''' <returns>Typowy obiekt List zawierajcy obiekty <see cref="Bug">Bug</see>.</returns>
  Public Function GetBugList(ByVal searchCriteria As SearchCriteria) As List(Of Bug)
    If _memberId = Guid.Empty Then
      Throw New ApplicationException("Nie mozna wywoa GetBugList bez ustawienia jej waciwoci Member. Przelij do konstruktora MemberId zanim wywoasz t metod")
    End If
    Return GetBugList("", searchCriteria)
  End Function

  ''' <summary>
  ''' Zwraca silnie typowan list wszystkich bedw posortowan wg <param name="sortExpression">sortExpression</param>
  ''' </summary>
  ''' <param name="sortExpression">Nazwa waciwoci obiektu <see cref="Bug">Bug</see>, wg ktrej lista ma by posortowana. W klasie <see cref="BugComparer">BugComparer</see> znajdziesz list prawidowych wartoci. T warto mona zakoczy sowem " desc", aby sortowa w kolejnoci malejcej.</param>
  ''' <returns>Typowa lista obiektw <see cref="Bug">Bug</see>.</returns>
  Public Function GetBugList(ByVal sortExpression As String, ByVal searchCriteria As SearchCriteria) As List(Of Bug)
    If _memberId = Guid.Empty Then
      Throw New ApplicationException("Nie mona wywoa GetBugList bez ustawienia waciwoci Member. Przelij do konstruktora MemberId zanim wywoasz t metod")
    End If

    _theBugList = BugManagerDB.GetBugList(searchCriteria, _memberId)

    ' Jeli lista ma wicej ni jedn pozycj, posortuj j.
    ' Jeli nie zdefiniowano sortExpression, obiekt Comparer wemie warto domyln Id desc.
    If _theBugList.Count > 1 Then
      _theBugList.Sort(New BugComparer(sortExpression))
    End If
    Return _theBugList
  End Function

  ''' <summary>
  ''' Zwraca obiekt Bug z bazy danych.
  ''' </summary>
  ''' <param name="id">Identyfikator bdu w bazie danych.</param>
  ''' <returns>W wypadku znalezienia identyfikatora wypeniony obiekt bdu, w przeciwnym wypadku null.</returns>
  Public Function GetBug(ByVal id As Integer) As Bug
    If _memberId = Guid.Empty Then
      Throw New ApplicationException("Nie mona wywoa metody GetBug bez przypisania wartoci waciwoci Member. Przelij do konstruktora MemberId zanim wywoasz t metod")
    End If
    Return BugManagerDB.GetBug(id, _memberId)
  End Function

  ''' <summary>
  ''' Tworzy nowy lub aktualizuje istniejcy bd
  ''' </summary>
  ''' <param name="theBug">Obiekt Bug, ktry ma by dodany lub aktualizowany.</param>
  ''' <remarks>Bd przesany do tej metody bedzie mie identyfikator wikszy od 0, jeli operacja si powiedzie.</remarks>
  Public Function InsertUpdateBug(ByVal theBug As Bug) As Integer
    If _memberId = Guid.Empty Then
      Throw New ApplicationException("Nie mona wywoa metody InsertUpdateBug bez przypisania wartoci waciwoci Member. Przelij do konstruktora MemberId zanim wywoasz t metod")
    End If
    BugManagerDB.InsertUpdateBug(theBug)
  End Function

#End Region

#Region "Klasy zagniedone"

  ''' <summary>
  ''' Klasa BugComparer, implementujca interfejs IComparer, jest uywana do sortowania bedw.
  ''' Jej metoda Compare dziaa dla kadej ze znaczcych waciwoci publicznych klasy Bug.
  ''' </summary>
  Public Class BugComparer
    Implements IComparer(Of Bug)

    Private _sortColumn As String ' Zawiera waciwo, wg ktrej mamy sortowa
    Private _reverse As Boolean

    ''' <summary>
    ''' Konstruktor klasy BugComparer. Ta metoda przyjmuje acuch sortExpression w formacie "NazwaWaciwoci" lub "NazwaWaciwoci desc", jeli lista ma by odwrotnie sortowana.
    ''' </summary>
    ''' <param name="sortExpression">Nazwa (publicznej) waciwoci klasy <see cref="Bug">Bug</see> wg ktrej bdziemy sortowa.</param>
    Public Sub New(ByVal sortExpression As String)
      If sortExpression.Length = 0 Then
        sortExpression = "Id desc" ' domylnie
      End If

      ' Sprawd czy mamy sortowa w porzdku malejcym
      _reverse = sortExpression.ToLowerInvariant().EndsWith(" desc")

      If _reverse Then
        ' Wytnij kocwk " desc"
        _sortColumn = sortExpression.Substring(0, sortExpression.Length - 5)
      Else
        _sortColumn = sortExpression
      End If
    End Sub

    ''' <summary>
    ''' Ta metoda przyjmuje jako argumenty dwa obiekty <see cref="Bug">Bug</see> i zwraca warto typu Integer wskazujc czy bd A jest mniejszy, rwny, czy wikszy ni bd B.
    ''' </summary>
    ''' <param name="a">Bd "lewostronny"</param>
    ''' <param name="b">Bd "prawostronny"</param>
    ''' <returns>Mniej ni 0 jeli bd a jest mniejszy ni b, zero jeli s jednakowe, a wiecej niz zero jeli bd a jest wikszy ni b.</returns>
    ''' <remarks>Jeli przesana do konstruktora warto sortColumn koczy si na " desc", rezulat jest odwrotny.</remarks>
    Public Function Compare(ByVal a As Bug, ByVal b As Bug) As Integer Implements IComparer(Of Bug).Compare
      Dim retVal As Integer = 0
      Select Case _sortColumn.ToLower()

        Case "id", ""
          retVal = a.Id.CompareTo(b.Id)
        Case "title"
          retVal = String.Compare(a.Title, b.Title, StringComparison.InvariantCultureIgnoreCase)
        Case "description"
          retVal = String.Compare(a.Description, b.Description, StringComparison.InvariantCultureIgnoreCase)
        Case "application"
          retVal = String.Compare(a.Application.Name, b.Application.Name, StringComparison.InvariantCultureIgnoreCase)
        Case "feature"
          retVal = String.Compare(a.Feature.Name, b.Feature.Name, StringComparison.InvariantCultureIgnoreCase)
        Case "frequency"
          retVal = String.Compare(a.Frequency.Name, b.Frequency.Name, StringComparison.InvariantCultureIgnoreCase)
        Case "severity"
          retVal = String.Compare(a.Severity.Name, b.Severity.Name, StringComparison.InvariantCultureIgnoreCase)
        Case "reproducibility"
          retVal = String.Compare(a.Reproducibility.Name, b.Reproducibility.Name, StringComparison.InvariantCultureIgnoreCase)
        Case "status"
          retVal = String.Compare(a.Status.Name, b.Status.Name, StringComparison.InvariantCultureIgnoreCase)
        Case "createddateandtime"
          retVal = DateTime.Compare(a.CreatedDateAndTime, b.CreatedDateAndTime)
        Case "updateddateandtime"
          retVal = DateTime.Compare(a.CreatedDateAndTime, b.CreatedDateAndTime)
      End Select

      Dim _reverseInt As Integer = 1
      If (_reverse) Then
        _reverseInt = -1
      End If
      Return (retVal * _reverseInt)
    End Function
  End Class
#End Region

End Class